# PineTrack

**A privacy-conscious, cookie-free tracking tool for recording ad conversions only after user consent.**

PineTrack helps ethical businesses measure the effectiveness of online advertising **without invasive tracking**, third-party scripts, or cookie banners. It records conversions **only if the user gives clear consent**, and stores the resulting data securely in a local CSV file for later upload to Google Ads or Microsoft Ads.

---

## 🌲 Introduction

PineTrack is a lightweight PHP-based tracking solution designed for ethically-minded course creators, small businesses, and developers who want to track ad conversions **without breaching privacy regulations**.

Unlike traditional tracking scripts that fire conversions via JavaScript or pixel events as soon as a page loads, **PineTrack tracks only after user consent**. It works without third-party cookies and does not send data to external servers — everything runs **locally on your server**.

---

## ✅ Why Use PineTrack?

- ✔️ **No third-party cookies** or external script dependencies  
- ✔️ **Fully GDPR-compliant** with explicit user consent required  
- ✔️ **Server-side data storage** (CSV) — no databases or trackers  
- ✔️ **Works with Google Ads & Microsoft Ads** offline conversion import  
- ✔️ **Great for ethical digital products, course funnels, and form-based sales**  
- ✔️ **Free and open**, just drop into your project and adjust the config

---

## ⚖️ Legal Considerations & Privacy Statement

**Legal basis:**  
PineTrack uses a **consent-based model** under **Art. 6 Abs. 1 lit. a DSGVO (EU GDPR)**. Tracking only occurs if the user actively agrees — either via a checkbox in the order form or via a dialog box on the thank you page.

**Privacy Statement Paragraph (German):**

> Wir messen den Erfolg unserer Online-Werbung auf datensparsame Weise. Wenn Sie uns auf der Bestellbestätigungsseite ausdrücklich Ihre Zustimmung erteilen, speichern wir anonymisierte Informationen zur Herkunft Ihres Besuchs (z. B. von einer Anzeige bei Google oder Microsoft). Diese Daten enthalten keine personenbezogenen Informationen und dienen ausschließlich der statistischen Auswertung und Optimierung unserer Werbeanzeigen.  
> Die Speicherung erfolgt auf Grundlage Ihrer freiwilligen Einwilligung gemäß Art. 6 Abs. 1 lit. a DSGVO. Sie können diese Einwilligung jederzeit für die Zukunft widerrufen. Die gespeicherten Daten werden automatisch nach 90 Tagen gelöscht.  
> Wir verwenden keine Cookies von Drittanbietern und verzichten auf den Einsatz externer Tracking-Skripte. Die Auswertung erfolgt vollständig intern.

---

## ⚙️ Technical Implementation

### Folder Structure

/pinetrack/
│
├── config.php # Path config for CSV directory
├── consent-handler.php # Main logic: processes consent and writes conversion
├── consent-ui.html # HTML for the thank-you page consent dialogue
├── prune-csv.php # Cleans up CSV files (deletes lines older than 90 days)
├── thankyou-consent.js # JS logic to load consent UI on thank-you page
└── data/ # Directory where .csv conversion files are stored

---

## 1. Parameter Tracking with Session

In your landing page or page where ads first arrive, store click IDs in the session:

<?php
session_start();

if (!empty($_GET['gclid'])) {
    $_SESSION['gclid'] = $_GET['gclid'];
}
if (!empty($_GET['msclkid'])) {
    $_SESSION['msclkid'] = $_GET['msclkid'];
}
?>

Add this early in your template or landing page PHP file.

## 2. Consent Collection at Conversion Point

### Method A: Consent Checkbox in Order Form

Add a required checkbox field to your checkout form:


<label>
  <input type="checkbox" name="consent_marketing" required>
  Ich erlaube die anonyme Zuordnung dieses Kaufs zur Anzeigenquelle.
</label>


Put this line into your php form submission script to capture the consent in the session:

// Consent-Flag
if ($consent) { $_SESSION['consent_marketing'] = 1; } else { unset($_SESSION['consent_marketing']); }

On the thank-you page (after successful submission), include this PHP snippet:

<?php include '/pinetrack/consent-handler.php'; ?>

Note: If you're using the JavaScript method (thankyou-consent.js), you do not need to include consent-handler.php manually on the thank-you page. The script handles all communication via AJAX.


### Method B: Consent Dialogue on Thank You Page

Include the following script at the end of your **thank you page**:

<script>
  window.pineConversionName = 'CourseSale'; // This is the name of your conversion goal as set in Google or Microsoft Ads. Defaults to "Lead"
  window.pineConversionValue = "0.00"; // This is the value of your conversion goal. Defaults to "0.00"
  window.pineConversionCurrency ="EUR"; // This is the currency of your conversion goal. Defaults to "EUR"
</script>
<script src="/pinetrack/thankyou-consent.js" defer></script>

This will:

1. Check for existing consent (for example given in the order form)
2. If none found, load the `consent-ui.html` HTML box to show the consent dialogue. If consent has already been given the dialogue is not being shown.
3. Upon user click, send consent + conversion name to `consent-handler.php` OR write the conversion silently if consent has already been given earlier

You can customize the dialogue in `/pinetrack/consent-ui.html`.

### ✅ Best Practice: Combine Checkbox and Thank You Page Dialogue

The recommended default implementation of PineTrack uses a **hybrid approach** that combines:

1. **A consent checkbox in the order form** *(optional but encouraged)
    → Captures marketing consent *upfront*, during checkout or lead form submission.
2. **A polite dialogue on the thank-you page**
    → Offers a **second, transparent opportunity** to confirm consent if it wasn't given earlier.

------

#### Why This Hybrid Approach?

- ✔️ **Redundancy without pressure**
   If the user skips the checkbox (by accident or intentionally), they can still opt in — *voluntarily* — after completing their action.
- ✔️ **Maximizes legitimate tracking**
   More users means better campaign data, without resorting to dark patterns or nudging.
- ✔️ **Graceful fallback**
   The thank-you dialogue only appears if no consent is found in the session — so users aren’t asked twice.
- ✔️ **User-friendly and transparent**
   The dialogue explains what’s being tracked, how long it's stored, and why — in simple language.

------

#### Example Flow

| Step                                           | What Happens                                  | PineTrack Behavior                                  |
| ---------------------------------------------- | --------------------------------------------- | --------------------------------------------------- |
| User submits form **with checkbox ticked**     | Consent is stored in session                  | Conversion is written silently via PHP, no UI shown |
| User submits form **without ticking checkbox** | No consent in session                         | Consent dialogue appears on thank-you page          |
| User clicks "Ja, erlauben" in dialogue         | Consent is stored + conversion written via JS | Box fades out politely after confirmation           |

------

#### Summary

> Combining the form checkbox **and** the thank-you page dialogue is the most robust, respectful, and conversion-friendly approach.
>  It works well for both **free lead magnets** and **paid digital products**, and ensures you're always **on the right side of GDPR**.

---

## 3. Conversion Writing and 90-Day Cleanup

### CSV Write Format

If the user has consented and a click ID is present, PineTrack writes a conversion to:

- `/data/gclids.csv` (Google Ads)
- `/data/msclkids.csv` (Microsoft Ads)

Format (Google Ads):

```
Google Click ID,Conversion Name,Conversion Time,Attributed Credit,Conversion Value,Conversion Currency
abc123456,Lead,2025-08-01T12:00:00Z,,0.00,EUR
```

Format (Microsoft Ads):

```
Microsoft Click ID,Conversion Name,Conversion Time,Conversion Value,Conversion Currency
abc123456,Lead,2025-08-01 12:00:00,,EUR
```

### 90-Day Cleanup

Add a scheduled cron job (e.g. daily) that runs:

```
php /path/to/prune-csv.php
```

This script removes lines older than 90 days while preserving headers and formatting.

---

## 4. Upload Offline Conversions

Once per week (or more frequently), download the CSV files and upload them manually — or set up an automated schedule:

- **Google Ads:**
  - Upload the CSV in your Google Ads account.
  - Specify the correct time zone and attribution model if prompted.
- **Microsoft Ads:**
  - Use the import function for offline conversions.

⚠️ *Always double-check that the column headers and formatting match the expected template before uploading.*

### 🔐 Secure Your Data Directory

For security reasons, we strongly recommend changing the data folder path in `config.php` to a location **outside the public web root** (e.g., not accessible via the browser).
 This ensures that your collected conversion data cannot be accessed directly via a URL.

---

## 🧾 Recap & Implementation Checklist

- [] Add session tracking for `gclid` and/or `msclkid`
-  [] Implement either consent method:
  -  [] Checkbox in order form (set session) | Add input field to form and consent flag to form submit script
  -  [] AND/OR JavaScript dialog on thank you page
-  [] Include `consent-handler.php` on the thank-you page if you use the checkbox only method without the dialogue script.
-  [] Customize `config.php` to set the secure CSV storage path
-  [] Review output file format for Google/Microsoft compatibility
-  [] Set up weekly cleanup via `prune-csv.php`
-  [] Manually upload CSVs to ad platforms or set up automatic upload schedule

---


## 📜 License

PineTrack is free to use for personal and commercial projects.

Attribution is not required, but always appreciated.
A short link to mediasequoia.com in your site notice or README goes a long way in supporting this project. Thank you!

---

## 🗂️ Version History

### 1.0 – Initial Release | 18.08.2025

- Core session-based tracking for `gclid` and `msclkid`
- Consent-based conversion logging via form checkbox or thank-you page dialogue
- CSV output matching Google & Microsoft offline import templates
- Prune script for automatic 90-day data retention
- Lightweight, dependency-free JS and PHP
- Configuration via `config.php` for flexible deployment


---

Built with ❤️ in the forest by an honest homesteader.