<?php
/**
 * PineTrack – Unified Consent Handler
 * 
 * Called via:
 * - AJAX from Thank You page (button click)
 * - Page load after form submission (if consent already in session)
 * 
 * If consent exists and a click ID is present, writes a conversion entry
 * using a dynamic conversion name passed via POST.
 */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if consent was given via session or just received via AJAX
$consentViaForm = !empty($_SESSION['consent_marketing']);
$consentViaAjax = !empty($_POST['consent_marketing']) && $_POST['consent_marketing'] === '1';

// If AJAX-based consent just arrived, store it in session
if ($consentViaAjax && !$consentViaForm) {
    $_SESSION['consent_marketing'] = 1;
    $consentViaForm = true;
}

// Read conversion name from POST, fallback to 'Lead'
$conversionName = 'Lead';
if (!empty($_POST['conversion'])) {
    $safeName = preg_replace('/[^A-Za-z0-9 _-]/', '', $_POST['conversion']);
    if (!empty($safeName)) {
        $conversionName = $safeName;
    }
}

// Read conversion value from POST, fallback to '0.00'
$conversionValue = '0.00';
if (!empty($_POST['conversion_value'])) {
    $safeValue = preg_replace('/[^0-9.,]/', '', $_POST['conversion_value']);
    $conversionValue = str_replace(',', '.', $safeValue); // normalize to dot
}

// Read conversion currency from POST, fallback to 'EUR'
$conversionCurrency = 'EUR';
if (!empty($_POST['conversion_currency'])) {
    $safeCurrency = preg_replace('/[^A-Z]/', '', strtoupper($_POST['conversion_currency']));
    if (!empty($safeCurrency)) {
        $conversionCurrency = $safeCurrency;
    }
}


// Proceed only if consent is confirmed
if ($consentViaForm) {
    $timestamp = gmdate('c'); // ISO 8601 UTC

    require_once __DIR__ . '/config.php';
    $csvDir = defined('PINETRACK_DATA_DIR') ? PINETRACK_DATA_DIR : (__DIR__ . '/data/');
    if (!is_dir($csvDir)) {
        mkdir($csvDir, 0775, true);
    }

    // GOOGLE ADS CSV
    if (!empty($_SESSION['gclid'])) {
        $gclidFile = $csvDir . "gclids.csv";
        $gclidHeader = "Parameters:TimeZone=+0100,,,,,\n" . "Google Click ID,Conversion Name,Conversion Time,Attributed Credit,Conversion Value,Conversion Currency\n";
        $gclidLine = implode(',', [
            $_SESSION['gclid'],
            $conversionName,
            $timestamp,
            '',
            $conversionValue,
            $conversionCurrency
        ]) . "\n";

        if (!file_exists($gclidFile)) {
            file_put_contents($gclidFile, $gclidHeader . $gclidLine, LOCK_EX);
        } else {
            file_put_contents($gclidFile, $gclidLine, FILE_APPEND | LOCK_EX);
        }
    }

    // MICROSOFT ADS CSV
    if (!empty($_SESSION['msclkid'])) {
        $msclkidFile = $csvDir . "msclkids.csv";
        $msclkidHeader = "Parameters:TimeZone=+0100,,,,\n" . "Microsoft Click ID,Conversion Name,Conversion Time,Conversion Value,Conversion Currency\n";
        $msclkidLine = implode(',', [
            $_SESSION['msclkid'],
            $conversionName,
            $timestamp,
            $conversionValue,
            $conversionCurrency
        ]) . "\n";

        if (!file_exists($msclkidFile)) {
            file_put_contents($msclkidFile, $msclkidHeader . $msclkidLine, LOCK_EX);
        } else {
            file_put_contents($msclkidFile, $msclkidLine, FILE_APPEND | LOCK_EX);
        }
    }

    // Clear session to prevent duplicate entries
    unset($_SESSION['consent_marketing'], $_SESSION['gclid'], $_SESSION['msclkid']);

    echo "Consent confirmed. Conversion written.";
    exit;
} else {
    http_response_code(204); // No content
    echo "No consent available.";
    exit;
}
