<?php
/**
 * PineTrack – Click ID Capture for Landing Pages
 * 
 * This script runs on all landing pages to securely capture
 * ad platform click IDs (e.g., gclid, msclkid) without using
 * cookies or localStorage.
 * 
 * The IDs are stored in the PHP session and will only be used
 * if the user later gives consent at the conversion point.
 */

// Start the session if it's not already active
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Define which click ID parameters we want to capture
$clickIdParameters = ['gclid', 'msclkid'];

/**
 * Validate a click ID:
 * - Only allow letters, numbers, dots, underscores, and hyphens
 * - Return empty string if invalid
 *
 * @param string $id
 * @return string
 */
function sanitizeClickId($id) {
    return preg_match('/^[A-Za-z0-9._-]+$/', $id) ? $id : '';
}

// Loop through each click ID parameter
foreach ($clickIdParameters as $paramName) {
    // Check if the parameter exists in the URL
    if (!empty($_GET[$paramName])) {
        $cleanId = sanitizeClickId($_GET[$paramName]);
        
        // Store the valid click ID in the session
        if ($cleanId !== '') {
            $_SESSION[$paramName] = $cleanId;
        }
    }
}
